<?php


namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Question;
use App\Models\Option;
use App\Models\TestAttempt;
use App\Models\Answer;
use App\Models\Formula;
use App\Models\FormulaResult;
use App\Calculators\FormulaCalculator;

class SetupTestResultsSeeder extends Seeder
{
    public function run()
    {
        $testId = 1;

        // پاک کردن داده‌های قدیمی
        FormulaResult::where('test_id', $testId)->delete();
        TestAttempt::where('test_id', $testId)->delete();

        // گرفتن تمام سوالات تست
        $questions = Question::where('test_id', $testId)
            ->with('options')
            ->orderBy('order')
            ->get();

        echo "✅ تعداد سوالات یافت شده: {$questions->count()}\n";

        if ($questions->count() < 3) {
            echo "❌ خطا: حداقل ۳ سوال نیاز است!\n";
            return;
        }

        // ساخت Attempt جدید
        $attempt = TestAttempt::create([
            'user_id' => 1,
            'test_id' => $testId,
            'participant_name' => 'نمونه کامل تست',
            'participant_phone' => '09120000000',
            'status' => 'completed',
            'total_score' => 0,
            'completed_at' => now(),
        ]);

        // پاسخ به هر سوال (گزینه با بیشترین امتیاز)
        $totalScore = 0;
        $answersData = [];
        $questionScores = [];

        foreach ($questions as $q) {
            $option = $q->options->sortByDesc('score_value')->first();
            if ($option) {
                $totalScore += $option->score_value;
                $questionScores[$q->id] = $option->score_value;
                $answersData[] = [
                    'attempt_id' => $attempt->id,
                    'question_id' => $q->id,
                    'option_id' => $option->id,
                    'text_answer' => null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
        }

        Answer::insert($answersData);
        $attempt->update(['total_score' => $totalScore]);

        echo "✅ Attempt ساخته شد: ID = {$attempt->id}\n";
        echo "✅ نمره کل: {$totalScore}\n";
        echo "✅ تعداد پاسخ‌ها: " . count($answersData) . "\n";

        // آپدیت فرمول‌ها با سوال‌های واقعی
        $formulas = Formula::where('test_id', $testId)->get();
        $questionIds = $questions->pluck('id')->toArray();

        foreach ($formulas as $formula) {
            $config = is_string($formula->formula_config) ? json_decode($formula->formula_config, true) : ($formula->formula_config ?? []);
            $newConfig = [];

            foreach ($config as $varName => $qId) {
                if (is_numeric($qId) && $qId > 0) {
                    // اگر question_id معتبر است، از اولین سوال استفاده کن
                    $newConfig[$varName] = $questionIds[0] ?? 1;
                } else {
                    $newConfig[$varName] = $qId;
                }
            }

            // برای فرمول‌های خاص، مقادیر خاص بده
            if ($formula->code == 'MATH_DIV' || $formula->code == 'MATH_MOD') {
                // برای تقسیم و باقی‌مانده، مطمئن شو Q1 صفر نیست
                $formula->update(['formula_config' => json_encode([
                    'Q1' => $questionIds[0] ?? 1,
                    'Q2' => $questionIds[1] ?? 2,
                ])]);
            } elseif ($formula->code == 'STR_CONTAINS_TRUE') {
                $formula->update(['formula_config' => json_encode([
                    'Answer' => 'من اضطراب دارم',
                ])]);
            } elseif ($formula->code == 'STR_CONTAINS_FALSE') {
                $formula->update(['formula_config' => json_encode([
                    'Answer' => 'من خوبم',
                ])]);
            } elseif ($formula->code == 'STR_LEN') {
                $formula->update(['formula_config' => json_encode([
                    'Answer' => 'سلام دنیا',
                ])]);
            }
        }

        // محاسبه فرمول‌ها
        $calculator = new FormulaCalculator();
        $formulas = Formula::where('test_id', $testId)->orderBy('priority')->get();
        $formulaResults = [];

        foreach ($formulas as $formula) {
            $config = is_string($formula->formula_config) ? json_decode($formula->formula_config, true) : ($formula->formula_config ?? []);
            $variables = [];

            foreach ($config as $varName => $value) {
                if (is_numeric($value) && $value > 0) {
                    // این یک question_id است
                    $answer = Answer::where('attempt_id', $attempt->id)
                        ->where('question_id', $value)
                        ->with('option')
                        ->first();
                    $variables[$varName] = $answer?->option?->score_value ?? 0;
                } else {
                    // این یک مقدار مستقیم است (رشته)
                    $variables[$varName] = $value;
                }
            }

            $calcResult = $calculator->calculate(
                $formula->formula_expression,
                $variables,
                $formulaResults
            );

            $formulaResults[$formula->code] = $calcResult;

            FormulaResult::create([
                'test_id' => $testId,
                'user_id' => 1,
                'attempt_id' => $attempt->id,
                'formula_id' => $formula->id,
                'formula_code' => $formula->code,
                'input_values' => $variables,
                'expression_snapshot' => $formula->formula_expression,
                'result_type' => $calcResult['type'] ?? 'unknown',
                'result_value' => $calcResult['value'] ?? null,
                'execution_time' => null,
            ]);

            echo "✅ فرمول {$formula->code}: {$calcResult['value']} " . ($calcResult['success'] ? '✅' : '❌') . "\n";
        }

        echo "\n🎉 تمام! حالا صفحه را باز کن:\n";
        echo "http://127.0.0.1:8000/tests/{$testId}/formula-results/{$attempt->id}\n";
    }
}
