<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Formula;
use App\Models\Test;

class FormulaTestSeeder extends Seeder
{
    public function run()
    {
        // پیدا کردن یک تست موجود برای اتصال فرمول‌ها
        $test = Test::first();
        if (!$test) {
            $this->command->error('❌ هیچ تستی در دیتابیس وجود ندارد!');
            return;
        }

        $this->command->info("✅ Using test: {$test->title} (ID: {$test->id})");

        // ═══════════════════════════════════════════════════
        // لیست کامل فرمول‌های تستی برای تمام دسته‌های عملگرها
        // ═══════════════════════════════════════════════════
        $formulas = [
            // ───────────────────────────────────────────────
            // ۱. عملگرهای ریاضی (Arithmetic Operators)
            // ───────────────────────────────────────────────
            [
                'name' => 'ریاضی: جمع ساده',
                'code' => 'MATH_ADD',
                'expression' => '{Q1} + {Q2}',
                'config' => ['Q1' => 1, 'Q2' => 2],
                'expected_result' => 3, // اگر Q1=1 و Q2=2 باشد
                'description' => 'تست عملگر جمع (+)'
            ],
            [
                'name' => 'ریاضی: تفریق',
                'code' => 'MATH_SUB',
                'expression' => '{Q1} - {Q2}',
                'config' => ['Q1' => 10, 'Q2' => 4],
                'expected_result' => 6,
                'description' => 'تست عملگر تفریق (-)'
            ],
            [
                'name' => 'ریاضی: ضرب',
                'code' => 'MATH_MUL',
                'expression' => '{Q1} * {Q2}',
                'config' => ['Q1' => 5, 'Q2' => 3],
                'expected_result' => 15,
                'description' => 'تست عملگر ضرب (*)'
            ],
            [
                'name' => 'ریاضی: تقسیم',
                'code' => 'MATH_DIV',
                'expression' => '{Q1} / {Q2}',
                'config' => ['Q1' => 20, 'Q2' => 4],
                'expected_result' => 5,
                'description' => 'تست عملگر تقسیم (/)'
            ],
            [
                'name' => 'ریاضی: باقی‌مانده',
                'code' => 'MATH_MOD',
                'expression' => '{Q1} % {Q2}',
                'config' => ['Q1' => 10, 'Q2' => 3],
                'expected_result' => 1,
                'description' => 'تست عملگر باقی‌مانده (%)'
            ],
            [
                'name' => 'ریاضی: توان',
                'code' => 'MATH_POW',
                'expression' => '{Q1} ^ 2',
                'config' => ['Q1' => 5],
                'expected_result' => 25,
                'description' => 'تست عملگر توان (^)'
            ],

            // ───────────────────────────────────────────────
            // ۲. عملگرهای مقایسه‌ای (Comparison Operators)
            // ───────────────────────────────────────────────
            [
                'name' => 'مقایسه: بزرگتر',
                'code' => 'CMP_GT',
                'expression' => '{Q1} > 5 ? 1 : 0',
                'config' => ['Q1' => 10],
                'expected_result' => 1,
                'description' => 'تست عملگر بزرگتر (>)'
            ],
            [
                'name' => 'مقایسه: کوچکتر',
                'code' => 'CMP_LT',
                'expression' => '{Q1} < 5 ? 1 : 0',
                'config' => ['Q1' => 3],
                'expected_result' => 1,
                'description' => 'تست عملگر کوچکتر (<)'
            ],
            [
                'name' => 'مقایسه: مساوی',
                'code' => 'CMP_EQ',
                'expression' => '{Q1} == 5 ? 1 : 0',
                'config' => ['Q1' => 5],
                'expected_result' => 1,
                'description' => 'تست عملگر مساوی (==)'
            ],
            [
                'name' => 'مقایسه: نامساوی',
                'code' => 'CMP_NEQ',
                'expression' => '{Q1} != 5 ? 1 : 0',
                'config' => ['Q1' => 3],
                'expected_result' => 1,
                'description' => 'تست عملگر نامساوی (!=)'
            ],
            [
                'name' => 'مقایسه: بزرگتر مساوی',
                'code' => 'CMP_GTE',
                'expression' => '{Q1} >= 5 ? 1 : 0',
                'config' => ['Q1' => 5],
                'expected_result' => 1,
                'description' => 'تست عملگر بزرگتر مساوی (>=)'
            ],
            [
                'name' => 'مقایسه: کوچکتر مساوی',
                'code' => 'CMP_LTE',
                'expression' => '{Q1} <= 5 ? 1 : 0',
                'config' => ['Q1' => 5],
                'expected_result' => 1,
                'description' => 'تست عملگر کوچکتر مساوی (<=)'
            ],

            // ───────────────────────────────────────────────
            // ۳. عملگرهای منطقی (Logical Operators)
            // ───────────────────────────────────────────────
            [
                'name' => 'منطقی: AND',
                'code' => 'LOG_AND',
                'expression' => '{Q1} > 5 && {Q2} < 10 ? 1 : 0',
                'config' => ['Q1' => 8, 'Q2' => 5],
                'expected_result' => 1,
                'description' => 'تست عملگر AND (&&)'
            ],
            [
                'name' => 'منطقی: OR',
                'code' => 'LOG_OR',
                'expression' => '{Q1} > 10 || {Q2} < 5 ? 1 : 0',
                'config' => ['Q1' => 3, 'Q2' => 3],
                'expected_result' => 1,
                'description' => 'تست عملگر OR (||)'
            ],
            [
                'name' => 'منطقی: NOT',
                'code' => 'LOG_NOT',
                'expression' => '!({Q1} > 10) ? 1 : 0',
                'config' => ['Q1' => 5],
                'expected_result' => 1,
                'description' => 'تست عملگر NOT (!)'
            ],

            // ───────────────────────────────────────────────
            // ۴. عملگر شرطی (Ternary Operator)
            // ───────────────────────────────────────────────
            [
                'name' => 'شرطی: ساده',
                'code' => 'TERN_SIMPLE',
                'expression' => '{Q1} > 5 ? {Q1} * 2 : {Q1}',
                'config' => ['Q1' => 10],
                'expected_result' => 20,
                'description' => 'تست عملگر شرطی ساده (شرط درست)'
            ],
            [
                'name' => 'شرطی: معکوس',
                'code' => 'TERN_FALSE',
                'expression' => '{Q1} > 5 ? {Q1} * 2 : {Q1}',
                'config' => ['Q1' => 3],
                'expected_result' => 3,
                'description' => 'تست عملگر شرطی (شرط غلط)'
            ],
            [
                'name' => 'شرطی: تودرتو',
                'code' => 'TERN_NESTED',
                'expression' => '{Q1} > 10 ? "High" : ({Q1} > 5 ? "Medium" : "Low")',
                'config' => ['Q1' => 7],
                'expected_result' => "Medium",
                'description' => 'تست عملگر شرطی تودرتو'
            ],

            // ───────────────────────────────────────────────
            // ۵. عملگرهای عضویت (Membership Operators)
            // ───────────────────────────────────────────────
            [
                'name' => 'عضویت: in (موجود)',
                'code' => 'MEM_IN_TRUE',
                'expression' => '{Q1} in [1, 3, 5] ? 1 : 0',
                'config' => ['Q1' => 3],
                'expected_result' => 1,
                'description' => 'تست عملگر in (مقدار موجود)'
            ],
            [
                'name' => 'عضویت: in (ناموجود)',
                'code' => 'MEM_IN_FALSE',
                'expression' => '{Q1} in [1, 3, 5] ? 1 : 0',
                'config' => ['Q1' => 10],
                'expected_result' => 0,
                'description' => 'تست عملگر in (مقدار ناموجود)'
            ],

            // ───────────────────────────────────────────────
            // ۶. عملگرهای رشته‌ای (String Operators)
            // ───────────────────────────────────────────────
            [
                'name' => 'رشته‌ای: contains (دارد)',
                'code' => 'STR_CONTAINS_TRUE',
                'expression' => "contains({Answer}, 'اضطراب') ? 1 : 0",
                'config' => ['Answer' => 1], // مقدار متنی در هنگام تست ست می‌شود
                'expected_result' => 1,
                'description' => 'تست تابع contains (پیدا شد)'
            ],
            [
                'name' => 'رشته‌ای: contains (ندارد)',
                'code' => 'STR_CONTAINS_FALSE',
                'expression' => "contains({Answer}, 'افسردگی') ? 1 : 0",
                'config' => ['Answer' => 1],
                'expected_result' => 0,
                'description' => 'تست تابع contains (پیدا نشد)'
            ],
            [
                'name' => 'رشته‌ای: طول رشته',
                'code' => 'STR_LEN',
                'expression' => 'len({Answer}) > 5 ? 1 : 0',
                'config' => ['Answer' => 1],
                'expected_result' => 1,
                'description' => 'تست تابع len'
            ],

            // ───────────────────────────────────────────────
            // ۷. توابع آماری (Statistical Functions)
            // ───────────────────────────────────────────────
            [
                'name' => 'آماری: میانگین',
                'code' => 'STAT_AVG',
                'expression' => 'avg({Q1}, {Q2}, {Q3})',
                'config' => ['Q1' => 10, 'Q2' => 20, 'Q3' => 30],
                'expected_result' => 20,
                'description' => 'تست تابع avg'
            ],
            [
                'name' => 'آماری: مجموع',
                'code' => 'STAT_SUM',
                'expression' => 'sum({Q1}, {Q2}, {Q3})',
                'config' => ['Q1' => 10, 'Q2' => 20, 'Q3' => 30],
                'expected_result' => 60,
                'description' => 'تست تابع sum'
            ],
            [
                'name' => 'آماری: حداقل',
                'code' => 'STAT_MIN',
                'expression' => 'min({Q1}, {Q2}, {Q3})',
                'config' => ['Q1' => 10, 'Q2' => 20, 'Q3' => 30],
                'expected_result' => 10,
                'description' => 'تست تابع min'
            ],
            [
                'name' => 'آماری: حداکثر',
                'code' => 'STAT_MAX',
                'expression' => 'max({Q1}, {Q2}, {Q3})',
                'config' => ['Q1' => 10, 'Q2' => 20, 'Q3' => 30],
                'expected_result' => 30,
                'description' => 'تست تابع max'
            ],

            // ───────────────────────────────────────────────
            // ۸. عملگر بین (Between Operator)
            // ───────────────────────────────────────────────
            [
                'name' => 'بین: در بازه',
                'code' => 'BETWEEN_TRUE',
                'expression' => '{Q1} between 5 and 15 ? 1 : 0',
                'config' => ['Q1' => 10],
                'expected_result' => 1,
                'description' => 'تست عملگر between (در بازه)'
            ],
            [
                'name' => 'بین: خارج بازه',
                'code' => 'BETWEEN_FALSE',
                'expression' => '{Q1} between 5 and 15 ? 1 : 0',
                'config' => ['Q1' => 20],
                'expected_result' => 0,
                'description' => 'تست عملگر between (خارج بازه)'
            ],

            // ───────────────────────────────────────────────
            // ۹. فرمول‌های ترکیبی پیچیده (Complex Combined)
            // ───────────────────────────────────────────────
            [
                'name' => 'ترکیبی: ریاضی + شرطی + منطقی',
                'code' => 'COMBO_1',
                'expression' => '({Q1} > 5 && {Q2} < 10) ? ({Q1} + {Q2}) * 2 : ({Q1} - {Q2})',
                'config' => ['Q1' => 8, 'Q2' => 5],
                'expected_result' => 26,
                'description' => 'تست ترکیب چند عملگر'
            ],
            [
                'name' => 'ترکیبی: آمار + شرطی + رشته',
                'code' => 'COMBO_2',
                'expression' => 'avg({Q1}, {Q2}) > 15 ? "High" : "Low"',
                'config' => ['Q1' => 20, 'Q2' => 30],
                'expected_result' => "High",
                'description' => 'تست ترکیب آمار و شرطی و رشته'
            ],
        ];

        // حذف فرمول‌های تستی قبلی (برای جلوگیری از تکرار)
        Formula::where('test_id', $test->id)
            ->whereIn('code', collect($formulas)->pluck('code'))
            ->delete();

        $this->command->info("\n📊 در حال ساخت " . count($formulas) . " فرمول تستی...\n");

        foreach ($formulas as $index => $formulaData) {
            Formula::create([
                'test_id' => $test->id,
                'name' => $formulaData['name'],
                'code' => $formulaData['code'],
                'formula_expression' => $formulaData['expression'],
                'formula_config' => json_encode($formulaData['config']),
                'type' => 'score',
                'priority' => $index,
                'is_active' => true,
                'variables' => $formulaData['config'],
                // ذخیره مقدار مورد انتظار برای تأیید صحت بعدی
                // (در یک ستون جداگانه یا جدول تست)
            ]);

            $this->command->info("  ✅ {$formulaData['code']}: {$formulaData['name']}");
        }

        $this->command->info("\n🎉 تمام فرمول‌های تستی با موفقیت ساخته شدند!\n");
        $this->command->info("📌 تعداد کل فرمول‌ها: " . count($formulas));
        $this->command->info("📌 تست ID: {$test->id}");
        $this->command->info("📌 برای اجرای تست از دستور زیر استفاده کن:");
        $this->command->info("   php artisan formula:test {$test->id}\n");
    }
}
